#!/bin/bash
set -euo pipefail

# ----- CONFIGURATION -----
INPUT_DIR="NC_000913.3_750_aligned_CDS_chunks"
FORMAT="png"
ROWS=2
COLUMNS=750
SRC_FILES=(main.c cdaig.c hopcroft.c)
CSV_NAME="results.csv"
# --------------------------

# Script directory
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"

# Convert INPUT_DIR to an absolute path
INPUT_DIR="${SCRIPT_DIR}/${INPUT_DIR}"

# ----- RESULTS DIRECTORY (robust) -----
last_exp=0
for d in "${SCRIPT_DIR}"/Experiment_*; do
    [ -d "$d" ] || continue
    num=$(basename "$d" | grep -oE '[0-9]+$')
    [ -n "$num" ] && (( num > last_exp )) && last_exp=$num
done

next_exp=$((last_exp + 1))
BASE_DIR="${SCRIPT_DIR}/Experiment_${next_exp}"
mkdir -p "$BASE_DIR"
echo "Created experiment folder: $BASE_DIR"

# ----- CSV INIT -----
CSV_FILE="${BASE_DIR}/${CSV_NAME}"
header="mutations,gaps,sequence_name,edges top,nodes top,suff top,edges bottom,nodes bottom,suff bottom,edges,nodes,suff"
echo "$header" > "$CSV_FILE"

# Enable nullglob
shopt -s nullglob

# ----- LOOP OVER FASTA FILES -----
for FASTA in "${INPUT_DIR}"/*.fasta; do
    [ -f "$FASTA" ] || continue
    FASTA_NAME="$(basename "$FASTA" .fasta)"
    SEQ_DIR="${BASE_DIR}/${FASTA_NAME}"
    mkdir -p "$SEQ_DIR"

    cp "$FASTA" "$SEQ_DIR/"
    BASE_FILE="$(basename "$FASTA")"
    echo ">>> Processing $FASTA_NAME"
    cd "$SEQ_DIR"

    # ---- Extract .txt files using fasta_extractor ----
    if [ -f "${SCRIPT_DIR}/fasta_extractor.c" ]; then
        cp "${SCRIPT_DIR}/fasta_extractor.c" .
        gcc -fsanitize=address -g -Wall -Wextra -o fasta_extractor fasta_extractor.c
        ./fasta_extractor "$BASE_FILE" 0

        txt_files=(*.txt)
        if [ ${#txt_files[@]} -ne 2 ]; then
            echo "Warning: expected 2 .txt files, found ${#txt_files[@]}"
            continue
        fi
    else
        echo "Warning: fasta_extractor.c not found"
        continue
    fi

    # Compile merger
    cp "${SCRIPT_DIR}/merge.c" .
    gcc -fsanitize=address -g -Wall -Wextra -o merger merge.c

    # Compile cdaig sources
    EXEC_DIR="${SEQ_DIR}/Exec"
    mkdir -p "$EXEC_DIR"
    for f in "${SRC_FILES[@]}"; do
        [ -f "${SCRIPT_DIR}/${f}" ] && cp "${SCRIPT_DIR}/${f}" "$EXEC_DIR/"
    done
    for h in "${SCRIPT_DIR}"/*.h; do
        [ -f "$h" ] && cp "$h" "$EXEC_DIR/"
    done
    cd "$EXEC_DIR"
    gcc -fsanitize=address -g -Wall -Wextra -o cdaig main.c cdaig.c hopcroft.c

    run_cdaig() {
        local file="$1"
        local output
        output=$(./cdaig "$file" $ROWS $COLUMNS)
        local v_val=$(echo "$output" | grep -oE "v=[0-9]+" | cut -d= -f2 || echo 0)
        local e_val=$(echo "$output" | grep -oE "e=[0-9]+" | cut -d= -f2 || echo 0)
        local suff_val=$(echo "$output" | grep -oE "suff=[0-9]+" | cut -d= -f2 || echo 0)
        echo "$e_val,$v_val,$suff_val"
    }

    # ---- Merge self1 with itself ----
    cd ..
    ./merger "${txt_files[0]}" "${txt_files[0]}" "self1_merged.txt"
    cp "self1_merged.txt" "$EXEC_DIR/"
    cd "$EXEC_DIR"
    self1_out=$(run_cdaig "self1_merged.txt")
    edges_top=$(echo "$self1_out" | cut -d, -f1)
    nodes_top=$(echo "$self1_out" | cut -d, -f2)
    suff_top=$(echo "$self1_out" | cut -d, -f3)

    # ---- Merge self2 with itself ----
    cd ..
    ./merger "${txt_files[1]}" "${txt_files[1]}" "self2_merged.txt"
    cp "self2_merged.txt" "$EXEC_DIR/"
    cd "$EXEC_DIR"
    self2_out=$(run_cdaig "self2_merged.txt")
    edges_bottom=$(echo "$self2_out" | cut -d, -f1)
    nodes_bottom=$(echo "$self2_out" | cut -d, -f2)
    suff_bottom=$(echo "$self2_out" | cut -d, -f3)

    # ---- Final merge of self1 + self2 ----
    cd ..
    ./merger "${txt_files[0]}" "${txt_files[1]}" "merged.txt"
    cp "merged.txt" "$EXEC_DIR/"
    cd "$EXEC_DIR"
    merged_out=$(run_cdaig "merged.txt")
    edges=$(echo "$merged_out" | cut -d, -f1)
    nodes=$(echo "$merged_out" | cut -d, -f2)
    suff=$(echo "$merged_out" | cut -d, -f3)

    # ----- Parse fasta name -----
    FNAME="${FASTA_NAME}"
    IFS='_' read -r MUT GAPS REST <<< "$FNAME"
    SEQ_NAME="${FNAME#${MUT}_${GAPS}_}"

    # ----- Write ONE row into CSV -----
    echo "$MUT,$GAPS,$SEQ_NAME,$edges_top,$nodes_top,$suff_top,$edges_bottom,$nodes_bottom,$suff_bottom,$edges,$nodes,$suff" >> "$CSV_FILE"

    cd "$BASE_DIR"
done

shopt -u nullglob

echo "All experiments completed. Results stored in $BASE_DIR"
echo "Summary CSV: $CSV_FILE"
